# Cargar las librerías necesarias
library(ade4)     # Para realizar el análisis de correspondencia
library(dplyr)    # Para manipulación de datos
library(ggplot2)  # Para visualización de datos

# Crear los datos en una tabla de contingencia (estratos vs percepciones)
datos_billete_cien <- data.frame(
  "Mucho" = c(50, 60, 80),
  "Poco" = c(50, 240, 50),
  "Nada" = c(300, 100, 70),
  row.names = c("Estratos1-2", "Estratos3-4", "Estratos5-6")
)

# Mostrar la tabla original para inspección
print(datos_billete_cien)

# Realizar un test de chi-cuadrado para ver si hay asociación entre filas y columnas
chisq.test(datos_billete_cien)

# Obtener el valor crítico de chi-cuadrado para un nivel de significancia del 5% con 4 grados de libertad
qchisq(0.95, 4)  # Esto ayuda a interpretar el resultado del test anterior

# Realizar el análisis de correspondencia (COA), extrayendo los dos primeros ejes (nf = 2)
res_ca <- dudi.coa(datos_billete_cien, scannf = FALSE, nf = 2)

# Calcular el porcentaje de varianza explicada por cada eje del análisis
porcentaje_varianza <- res_ca$eig / sum(res_ca$eig) * 100
round(porcentaje_varianza, 2)  # Redondear a 2 decimales

# Extraer coordenadas de las filas (estratos) del análisis
filas <- as.data.frame(res_ca$li) %>%
  mutate(Tipo = "Estrato",        # Añadir columna indicando que son estratos
         Label = rownames(res_ca$li))  # Añadir etiquetas con los nombres de fila

# Extraer coordenadas de las columnas (percepciones)
columnas <- as.data.frame(res_ca$co) %>%
  rename(Axis1 = Comp1, Axis2 = Comp2) %>%  # Renombrar columnas para graficar
  mutate(Tipo = "Percepción",              # Añadir columna indicando que son percepciones
         Label = rownames(res_ca$co))      # Añadir etiquetas con los nombres de columna

# Unir coordenadas de filas y columnas en un solo dataframe para graficar
biplot_data <- bind_rows(filas, columnas)
biplot_data  # Mostrar los datos combinados para el gráfico

# Crear un gráfico sin etiquetas (solo puntos de estratos y percepciones)
ggplot(biplot_data, aes(x = Axis1, y = Axis2, color = Tipo, shape = Tipo)) +
  geom_point(size = 4) +
  theme_minimal() +
  labs(title = "Análisis de Correspondencia",
       x = "Eje 1", y = "Eje 2", color = "Categoría", shape = "Categoría") +
  theme(legend.position = "right")

# Repetimos pasos para un gráfico con etiquetas
# Volvemos a preparar los datos (opcional si ya se tiene arriba)
filas <- as.data.frame(res_ca$li)
filas$Tipo <- "Estrato"
filas$Label <- rownames(filas)

columnas <- as.data.frame(res_ca$co)
colnames(columnas)[1:2] <- c("Axis1", "Axis2")
columnas$Tipo <- "Percepción"
columnas$Label <- rownames(columnas)

# Combinar filas y columnas con sus coordenadas para graficar
biplot_data <- rbind(filas[, c("Axis1", "Axis2", "Tipo", "Label")],
                     columnas[, c("Axis1", "Axis2", "Tipo", "Label")])

# Crear gráfico final con etiquetas en cada punto
ggplot(biplot_data, aes(x = Axis1, y = Axis2, color = Tipo, shape = Tipo)) +
  geom_point(size = 4) +  # Dibujar puntos
  geom_text(aes(label = Label), size = 3, vjust = -1, show.legend = FALSE) +  # Añadir etiquetas a cada punto
  theme_minimal() +
  labs(title = "Análisis de Correspondencia",
       x = "Eje 1", y = "Eje 2", color = "Categoría", shape = "Categoría") +
  theme(legend.position = "right")
